# Copyright 2013 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS-IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Cleans up ssh keys placed in GCS for distribution to datanodes, since the datanodes
# should have already downloaded and configured the ssh keys by now.
# Starts relevant hadoop daemon servers as the 'hadoop' user.

set -e

PRIVATE_KEY_NAME='hadoop_master_id_rsa'
PUBLIC_KEY_NAME="${PRIVATE_KEY_NAME}.pub"
REMOTE_PUBLIC_KEY="gs://${CONFIGBUCKET}/${NAMENODE_HOSTNAME}/${PUBLIC_KEY_NAME}"
gsutil rm ${REMOTE_PUBLIC_KEY}

cd ${HADOOP_INSTALL_DIR}

# Test for sshability to workers.
for NODE in ${WORKERS[@]}; do
  sudo -u hadoop ssh ${NODE} "exit 0"
done

if (( ${ENABLE_HDFS} )); then
  # Hadoop versions before 1.1 require hdfs to be the default filesystem to
  # format the namenode
  CONF_BACKUP=''
  HADOOP_VERSION="$(./bin/hadoop version | head -n 1 | sed 's/[^0-9.]//g')"
  if [[ "${DEFAULT_FS}" != hdfs ]] && [ "$(echo -e "${HADOOP_VERSION}\n1.1" \
      | sort -nrt . -k 1,1 -k 2,2 | head -n 1)" == '1.1' ]; then
    # Hadoop version < 1.1

    # Back up configuration
    CONF_BACKUP=/tmp/conf-backup-$(date +%s)
    cp -r ${HADOOP_CONF_DIR} ${CONF_BACKUP}

    NAMENODE_URI="hdfs://${NAMENODE_HOSTNAME}:8020/"
    ghconfig set_default_fs \
        --hadoop_conf_dir ${HADOOP_CONF_DIR} \
        --default_fs hdfs \
        --namenode_uri ${NAMENODE_URI}
    chown hadoop:hadoop ${HADOOP_CONF_DIR}
  fi

  if [ -z "$(find /mnt/*/hadoop/ /hadoop/ -maxdepth 4 \
      -wholename '*/name/current/VERSION')" ]; then
    # namenode is not formatted
    sudo -u hadoop ./bin/hadoop namenode -format
  fi

  sudo -u hadoop ./bin/start-dfs.sh

  # Switch back to orginal configuration
  if [[ -n "${CONF_BACKUP}" ]] && [[ -d "${CONF_BACKUP}" ]]; then
    rm -rf ${HADOOP_CONF_DIR}
    cp -r ${CONF_BACKUP} ${HADOOP_CONF_DIR}
    chown -R hadoop:hadoop ${HADOOP_CONF_DIR}
  fi
fi

# Start up job and task trackers
sudo -u hadoop ./bin/start-mapred.sh
