// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <stddef.h>
#include <memory>

#include "base/bind.h"
#include "base/callback.h"
#include "base/files/file_path.h"
#include "base/path_service.h"
#include "base/run_loop.h"
#include "chrome/browser/browsing_data/browsing_data_filter_builder.h"
#include "chrome/browser/browsing_data/browsing_data_helper.h"
#include "chrome/browser/browsing_data/browsing_data_remover.h"
#include "chrome/browser/browsing_data/browsing_data_remover_factory.h"
#include "chrome/browser/browsing_data/browsing_data_remover_test_util.h"
#include "chrome/browser/browsing_data/cache_counter.h"
#include "chrome/browser/chrome_notification_types.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/tabs/tab_strip_model.h"
#include "chrome/common/pref_names.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "chrome/test/base/ui_test_utils.h"
#include "components/browsing_data/core/browsing_data_utils.h"
#include "components/prefs/pref_service.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/download_manager.h"
#include "content/public/browser/web_contents.h"
#include "content/public/common/content_paths.h"
#include "content/public/test/browser_test_utils.h"
#include "content/public/test/download_test_observer.h"
#include "net/dns/mock_host_resolver.h"
#include "net/http/transport_security_state.h"
#include "net/test/embedded_test_server/embedded_test_server.h"
#include "net/url_request/url_request_context.h"
#include "net/url_request/url_request_context_getter.h"
#include "testing/gtest/include/gtest/gtest.h"

using content::BrowserThread;

namespace {
static const char* kExampleHost = "example.com";
}

class BrowsingDataRemoverBrowserTest : public InProcessBrowserTest {
 public:
  BrowsingDataRemoverBrowserTest() {}

  void SetUpOnMainThread() override {
    base::FilePath path;
    PathService::Get(content::DIR_TEST_DATA, &path);
    host_resolver()->AddRule(kExampleHost, "127.0.0.1");
    embedded_test_server()->ServeFilesFromDirectory(path);
    ASSERT_TRUE(embedded_test_server()->Start());
  }

  void RunScriptAndCheckResult(const std::string& script,
                               const std::string& result) {
    std::string data;
    ASSERT_TRUE(content::ExecuteScriptAndExtractString(
        browser()->tab_strip_model()->GetActiveWebContents(), script, &data));
    ASSERT_EQ(data, result);
  }

  void VerifyDownloadCount(size_t expected) {
    content::DownloadManager* download_manager =
        content::BrowserContext::GetDownloadManager(browser()->profile());
    std::vector<content::DownloadItem*> downloads;
    download_manager->GetAllDownloads(&downloads);
    EXPECT_EQ(expected, downloads.size());
  }

  void DownloadAnItem() {
    base::ScopedTempDir downloads_directory;
    ASSERT_TRUE(downloads_directory.CreateUniqueTempDir());
    browser()->profile()->GetPrefs()->SetFilePath(
        prefs::kDownloadDefaultDirectory, downloads_directory.GetPath());

    // Start a download.
    content::DownloadManager* download_manager =
        content::BrowserContext::GetDownloadManager(browser()->profile());
    std::unique_ptr<content::DownloadTestObserver> observer(
        new content::DownloadTestObserverTerminal(
            download_manager, 1,
            content::DownloadTestObserver::ON_DANGEROUS_DOWNLOAD_ACCEPT));

    GURL download_url = ui_test_utils::GetTestUrl(
        base::FilePath().AppendASCII("downloads"),
        base::FilePath().AppendASCII("a_zip_file.zip"));
    ui_test_utils::NavigateToURL(browser(), download_url);
    observer->WaitForFinished();

    VerifyDownloadCount(1u);
  }

  browsing_data::BrowsingDataCounter::ResultInt GetCacheSize() {
    base::RunLoop run_loop;
    browsing_data::BrowsingDataCounter::ResultInt size;

    Profile* profile = browser()->profile();
    CacheCounter counter(profile);
    counter.Init(profile->GetPrefs(),
                 base::Bind(&BrowsingDataRemoverBrowserTest::OnCacheSizeResult,
                            base::Unretained(this), base::Unretained(&run_loop),
                            base::Unretained(&size)));
    counter.Restart();
    run_loop.Run();
    return size;
  }

  void RemoveAndWait(int remove_mask) {
    BrowsingDataRemover* remover =
        BrowsingDataRemoverFactory::GetForBrowserContext(browser()->profile());
    BrowsingDataRemoverCompletionObserver completion_observer(remover);
    remover->RemoveAndReply(
        base::Time(), base::Time::Max(), remove_mask,
        BrowsingDataHelper::UNPROTECTED_WEB, &completion_observer);
    completion_observer.BlockUntilCompletion();
  }

  void RemoveWithFilterAndWait(
      int remove_mask,
      std::unique_ptr<BrowsingDataFilterBuilder> filter_builder) {
    BrowsingDataRemover* remover =
        BrowsingDataRemoverFactory::GetForBrowserContext(browser()->profile());
    BrowsingDataRemoverCompletionObserver completion_observer(remover);
    remover->RemoveWithFilterAndReply(
        base::Time(), base::Time::Max(), remove_mask,
        BrowsingDataHelper::UNPROTECTED_WEB, std::move(filter_builder),
        &completion_observer);
    completion_observer.BlockUntilCompletion();
  }

 private:
  void OnCacheSizeResult(
      base::RunLoop* run_loop,
      browsing_data::BrowsingDataCounter::ResultInt* out_size,
      std::unique_ptr<browsing_data::BrowsingDataCounter::Result> result) {
    if (!result->Finished())
      return;

    *out_size =
        static_cast<browsing_data::BrowsingDataCounter::FinishedResult*>(
            result.get())
            ->Value();
    run_loop->Quit();
  }
};

class BrowsingDataRemoverTransportSecurityStateBrowserTest
    : public BrowsingDataRemoverBrowserTest {
 public:
  BrowsingDataRemoverTransportSecurityStateBrowserTest() {}

  void SetUpOnMainThread() override {
    BrowserThread::PostTask(
        BrowserThread::IO, FROM_HERE,
        base::Bind(&BrowsingDataRemoverTransportSecurityStateBrowserTest::
                       SetUpTransportSecurityState,
                   base::Unretained(this),
                   base::RetainedRef(
                       browser()->profile()->GetRequestContext())));
  }

  void CheckTransportSecurityState(
      scoped_refptr<net::URLRequestContextGetter> context_getter,
      bool should_be_cleared) {
    net::TransportSecurityState* state =
        context_getter->GetURLRequestContext()->transport_security_state();
    if (should_be_cleared)
      EXPECT_FALSE(state->ShouldUpgradeToSSL("example.test"));
    else
      EXPECT_TRUE(state->ShouldUpgradeToSSL("example.test"));
  }

 protected:
  void SetUpTransportSecurityState(
      scoped_refptr<net::URLRequestContextGetter> context_getter) {
    net::TransportSecurityState* state =
        context_getter->GetURLRequestContext()->transport_security_state();
    base::Time expiry = base::Time::Now() + base::TimeDelta::FromDays(1000);
    EXPECT_FALSE(state->ShouldUpgradeToSSL("example.test"));
    state->AddHSTS("example.test", expiry, false);
    EXPECT_TRUE(state->ShouldUpgradeToSSL("example.test"));
  }
};

// Test BrowsingDataRemover for downloads.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverBrowserTest, Download) {
  DownloadAnItem();
  RemoveAndWait(BrowsingDataRemover::REMOVE_DOWNLOADS);
  VerifyDownloadCount(0u);
}

// The call to Remove() should crash in debug (DCHECK), but the browser-test
// process model prevents using a death test.
#if defined(NDEBUG) && !defined(DCHECK_ALWAYS_ON)
// Test BrowsingDataRemover for prohibited downloads. Note that this only
// really exercises the code in a Release build.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverBrowserTest, DownloadProhibited) {
  PrefService* prefs = browser()->profile()->GetPrefs();
  prefs->SetBoolean(prefs::kAllowDeletingBrowserHistory, false);

  DownloadAnItem();
  RemoveAndWait(BrowsingDataRemover::REMOVE_DOWNLOADS);
  VerifyDownloadCount(1u);
}
#endif

// Verify can modify database after deleting it.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverBrowserTest, Database) {
  GURL url = embedded_test_server()->GetURL("/simple_database.html");
  LOG(ERROR) << url;
  ui_test_utils::NavigateToURL(browser(), url);

  RunScriptAndCheckResult("createTable()", "done");
  RunScriptAndCheckResult("insertRecord('text')", "done");
  RunScriptAndCheckResult("getRecords()", "text");

  RemoveAndWait(BrowsingDataRemover::REMOVE_SITE_DATA);

  ui_test_utils::NavigateToURL(browser(), url);
  RunScriptAndCheckResult("createTable()", "done");
  RunScriptAndCheckResult("insertRecord('text2')", "done");
  RunScriptAndCheckResult("getRecords()", "text2");
}

// Verify that cache deleting cache finishes successfully. Complete deletion
// of cache should leave it empty, and partial deletion should leave nonzero
// amount of data. Note that this tests the integration of BrowsingDataRemover
// with ConditionalCacheDeletionHelper. Whether ConditionalCacheDeletionHelper
// actually deletes the correct entries is tested
// in ConditionalCacheDeletionHelperBrowsertest.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverBrowserTest, Cache) {
  // Load several resources.
  GURL url1 = embedded_test_server()->GetURL("/simple.html");
  GURL url2 = embedded_test_server()->GetURL(kExampleHost, "/simple.html");
  ASSERT_FALSE(url::IsSameOriginWith(url1, url2));
  ui_test_utils::NavigateToURL(browser(), url1);
  ui_test_utils::NavigateToURL(browser(), url2);

  // The cache is nonempty, because we created entries by visiting websites.
  browsing_data::BrowsingDataCounter::ResultInt original_size = GetCacheSize();
  EXPECT_GT(original_size, 0);

  // Partially delete cache data. Delete data for localhost, which is the origin
  // of |url1|, but not for |kExampleHost|, which is the origin of |url2|.
  std::unique_ptr<BrowsingDataFilterBuilder> filter_builder =
      BrowsingDataFilterBuilder::Create(BrowsingDataFilterBuilder::WHITELIST);
  filter_builder->AddOrigin(url::Origin(url1));
  RemoveWithFilterAndWait(BrowsingDataRemover::REMOVE_CACHE,
                          std::move(filter_builder));

  // After the partial deletion, the cache should be smaller but still nonempty.
  browsing_data::BrowsingDataCounter::ResultInt new_size = GetCacheSize();
  EXPECT_LT(new_size, original_size);

  // Another partial deletion with the same filter should have no effect.
  filter_builder =
      BrowsingDataFilterBuilder::Create(BrowsingDataFilterBuilder::WHITELIST);
  filter_builder->AddOrigin(url::Origin(url1));
  RemoveWithFilterAndWait(BrowsingDataRemover::REMOVE_CACHE,
                          std::move(filter_builder));
  EXPECT_EQ(new_size, GetCacheSize());

  // Delete the remaining data.
  RemoveAndWait(BrowsingDataRemover::REMOVE_CACHE);

  // The cache is empty.
  EXPECT_EQ(0, GetCacheSize());
}

// Verify that TransportSecurityState data is cleared for REMOVE_CACHE.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverTransportSecurityStateBrowserTest,
                       ClearTransportSecurityState) {
  RemoveAndWait(BrowsingDataRemover::REMOVE_CACHE);
  base::RunLoop run_loop;
  BrowserThread::PostTaskAndReply(
      BrowserThread::IO, FROM_HERE,
      base::Bind(&BrowsingDataRemoverTransportSecurityStateBrowserTest::
                     CheckTransportSecurityState,
                 base::Unretained(this),
                 base::RetainedRef(browser()->profile()->GetRequestContext()),
                 true /* should be cleared */),
      run_loop.QuitClosure());
}

// Verify that TransportSecurityState data is not cleared if REMOVE_CACHE is not
// set.
IN_PROC_BROWSER_TEST_F(BrowsingDataRemoverTransportSecurityStateBrowserTest,
                       PreserveTransportSecurityState) {
  RemoveAndWait(BrowsingDataRemover::REMOVE_SITE_DATA);
  base::RunLoop run_loop;
  BrowserThread::PostTaskAndReply(
      BrowserThread::IO, FROM_HERE,
      base::Bind(&BrowsingDataRemoverTransportSecurityStateBrowserTest::
                     CheckTransportSecurityState,
                 base::Unretained(this),
                 base::RetainedRef(browser()->profile()->GetRequestContext()),
                 false /* should not be cleared */),
      run_loop.QuitClosure());
}
