// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview 'settings-credit-card-edit-dialog' is the dialog that allows
 * editing or creating a credit card entry.
 */

(function() {
'use strict';

Polymer({
  is: 'settings-credit-card-edit-dialog',

  properties: {
    /**
     * The credit card being edited.
     * @type {!chrome.autofillPrivate.CreditCardEntry}
     */
    creditCard: Object,

    /**
     * The actual title that's used for this dialog. Will be context sensitive
     * based on if |creditCard| is being created or edited.
     * @private
     */
    title_: String,

    /**
     * The list of months to show in the dropdown.
     * @private {!Array<string>}
     */
    monthList_: {
      type: Array,
      value: [
        '01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12',
      ],
    },

    /**
     * The list of years to show in the dropdown.
     * @private {!Array<string>}
     */
    yearList_: Array,
  },

  behaviors: [
    I18nBehavior,
  ],

  /** @override */
  attached: function() {
    this.title_ = this.i18n(
        this.creditCard.guid ? 'editCreditCardTitle' : 'addCreditCardTitle');

    // Add a leading '0' if a month is 1 char.
    if (this.creditCard.expirationMonth.length == 1)
      this.creditCard.expirationMonth = '0' + this.creditCard.expirationMonth;

    var date = new Date();
    var firstYear = date.getFullYear();
    var lastYear = firstYear + 9;  // Show next 9 years (10 total).
    var selectedYear = parseInt(this.creditCard.expirationYear, 10);

    // |selectedYear| must be valid and between first and last years.
    if (!selectedYear)
      selectedYear = firstYear;
    else if (selectedYear < firstYear)
      firstYear = selectedYear;
    else if (selectedYear > lastYear)
      lastYear = selectedYear;

    var yearList = [];
    for (var i = firstYear; i <= lastYear; ++i) {
      yearList.push(i.toString());
    }
    this.yearList_ = yearList;

    this.expirationYear = this.creditCard.expirationYear;
    this.expirationMonth = this.creditCard.expirationMonth;

    this.$.dialog.showModal();
  },

  /** Closes the dialog. */
  close: function() {
    this.$.dialog.close();
  },

  /**
   * Handler for tapping the 'cancel' button. Should just dismiss the dialog.
   * @private
   */
  onCancelButtonTap_: function() {
    this.$.dialog.cancel();
  },

  /**
   * Handler for tapping the save button.
   * @private
   */
  onSaveButtonTap_: function() {
    this.creditCard.expirationYear = this.expirationYear;
    this.creditCard.expirationMonth = this.expirationMonth;
    this.fire('save-credit-card', this.creditCard);
    this.close();
  },
});
})();
