// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

Polymer({
  is: 'history-list-container',

  properties: {
    // The path of the currently selected page.
    selectedPage_: String,

    // Whether domain-grouped history is enabled.
    grouped: Boolean,

    /** @type {!QueryState} */
    queryState: Object,

    /** @type {!QueryResult} */
    queryResult: Object,
  },

  observers: [
    'groupedRangeChanged_(queryState.range)',
  ],

  listeners: {
    'history-list-scrolled': 'closeMenu_',
    'load-more-history': 'loadMoreHistory_',
    'toggle-menu': 'toggleMenu_',
  },

  /**
   * @param {HistoryQuery} info An object containing information about the
   *    query.
   * @param {!Array<HistoryEntry>} results A list of results.
   */
  historyResult: function(info, results) {
    this.initializeResults_(info, results);
    this.closeMenu_();

    if (this.selectedPage_ == 'grouped-list') {
      this.$$('#grouped-list').historyData = results;
      return;
    }

    var list = /** @type {HistoryListElement} */(this.$['infinite-list']);
    list.addNewResults(results, this.queryState.incremental);
    if (info.finished)
      list.disableResultLoading();
  },

  /**
   * Queries the history backend for results based on queryState.
   * @param {boolean} incremental Whether the new query should continue where
   *    the previous query stopped.
   */
  queryHistory: function(incremental) {
    var queryState = this.queryState;
    // Disable querying until the first set of results have been returned. If
    // there is a search, query immediately to support search query params from
    // the URL.
    var noResults = !this.queryResult || this.queryResult.results == null;
    if (queryState.queryingDisabled ||
        (!this.queryState.searchTerm && noResults)) {
      return;
    }

    // Close any open dialog if a new query is initiated.
    var dialog = this.$.dialog.getIfExists();
    if (!incremental && dialog && dialog.open)
      dialog.close();

    this.set('queryState.querying', true);
    this.set('queryState.incremental', incremental);

    var lastVisitTime = 0;
    if (incremental) {
      var lastVisit = this.queryResult.results.slice(-1)[0];
      lastVisitTime = lastVisit ? lastVisit.time : 0;
    }

    var maxResults =
      queryState.range == HistoryRange.ALL_TIME ? RESULTS_PER_PAGE : 0;
    chrome.send('queryHistory', [
      queryState.searchTerm, queryState.groupedOffset, queryState.range,
      lastVisitTime, maxResults
    ]);
  },

  historyDeleted: function() {
    // Do not reload the list when there are items checked.
    if (this.getSelectedItemCount() > 0)
      return;

    // Reload the list with current search state.
    this.queryHistory(false);
  },

  /** @return {number} */
  getSelectedItemCount: function() {
    return this.getSelectedList_().selectedPaths.size;
  },

  unselectAllItems: function(count) {
    var selectedList = this.getSelectedList_();
    if (selectedList)
      selectedList.unselectAllItems(count);
  },

  /**
   * Delete all the currently selected history items. Will prompt the user with
   * a dialog to confirm that the deletion should be performed.
   */
  deleteSelectedWithPrompt: function() {
    if (!loadTimeData.getBoolean('allowDeletingHistory'))
      return;
    this.$.dialog.get().then(function(dialog) {
      dialog.showModal();
    });
  },

  /**
   * @param {HistoryRange} range
   * @private
   */
  groupedRangeChanged_: function(range) {
    this.selectedPage_ = this.queryState.range == HistoryRange.ALL_TIME ?
      'infinite-list' : 'grouped-list';

    this.queryHistory(false);
  },

  /** @private */
  loadMoreHistory_: function() { this.queryHistory(true); },

  /**
   * @param {HistoryQuery} info
   * @param {!Array<HistoryEntry>} results
   * @private
   */
  initializeResults_: function(info, results) {
    if (results.length == 0)
      return;

    var currentDate = results[0].dateRelativeDay;

    for (var i = 0; i < results.length; i++) {
      // Sets the default values for these fields to prevent undefined types.
      results[i].selected = false;
      results[i].readableTimestamp =
          info.term == '' ? results[i].dateTimeOfDay : results[i].dateShort;

      if (results[i].dateRelativeDay != currentDate) {
        currentDate = results[i].dateRelativeDay;
      }
    }
  },

  /** @private */
  onDialogConfirmTap_: function() {
    this.getSelectedList_().deleteSelected();
    var dialog = assert(this.$.dialog.getIfExists());
    dialog.close();
  },

  /** @private */
  onDialogCancelTap_: function() {
    var dialog = assert(this.$.dialog.getIfExists());
    dialog.close();
  },

  /**
   * Closes the overflow menu.
   * @private
   */
  closeMenu_: function() {
    var menu = this.$.sharedMenu.getIfExists();
    if (menu)
      menu.closeMenu();
  },

  /**
   * Opens the overflow menu unless the menu is already open and the same button
   * is pressed.
   * @param {{detail: {item: !HistoryEntry, target: !HTMLElement}}} e
   * @return {Promise<Element>}
   * @private
   */
  toggleMenu_: function(e) {
    var target = e.detail.target;
    return this.$.sharedMenu.get().then(function(menu) {
      /** @type {CrSharedMenuElement} */(menu).toggleMenu(
        target, e.detail);
    });
  },

  /** @private */
  onMoreFromSiteTap_: function() {
    var menu = assert(this.$.sharedMenu.getIfExists());
    this.fire('search-domain', {domain: menu.itemData.item.domain});
    menu.closeMenu();
  },

  /** @private */
  onRemoveFromHistoryTap_: function() {
    var menu = assert(this.$.sharedMenu.getIfExists());
    var itemData = menu.itemData;
    md_history.BrowserService.getInstance()
        .deleteItems([itemData.item])
        .then(function(items) {
          this.getSelectedList_().removeItemsByPath([itemData.path]);
          // This unselect-all is to reset the toolbar when deleting a selected
          // item. TODO(tsergeant): Make this automatic based on observing list
          // modifications.
          this.fire('unselect-all');
        }.bind(this));
    menu.closeMenu();
  },

  /**
   * @return {HTMLElement}
   * @private
   */
  getSelectedList_: function() {
    return this.$.content.selectedItem;
  },
});
