// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ntp_snippets/content_suggestions_service_factory.h"

#include "base/feature_list.h"
#include "base/files/file_path.h"
#include "base/memory/ptr_util.h"
#include "base/memory/singleton.h"
#include "chrome/browser/bookmarks/bookmark_model_factory.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/search/suggestions/image_decoder_impl.h"
#include "chrome/browser/search/suggestions/suggestions_service_factory.h"
#include "chrome/browser/signin/profile_oauth2_token_service_factory.h"
#include "chrome/browser/signin/signin_manager_factory.h"
#include "chrome/common/channel_info.h"
#include "components/bookmarks/browser/bookmark_model.h"
#include "components/image_fetcher/image_decoder.h"
#include "components/image_fetcher/image_fetcher.h"
#include "components/image_fetcher/image_fetcher_impl.h"
#include "components/keyed_service/content/browser_context_dependency_manager.h"
#include "components/ntp_snippets/bookmarks/bookmark_suggestions_provider.h"
#include "components/ntp_snippets/content_suggestions_service.h"
#include "components/ntp_snippets/features.h"
#include "components/ntp_snippets/ntp_snippets_constants.h"
#include "components/ntp_snippets/ntp_snippets_database.h"
#include "components/ntp_snippets/ntp_snippets_fetcher.h"
#include "components/ntp_snippets/ntp_snippets_scheduler.h"
#include "components/ntp_snippets/ntp_snippets_service.h"
#include "components/ntp_snippets/ntp_snippets_status_service.h"
#include "components/prefs/pref_service.h"
#include "components/safe_json/safe_json_parser.h"
#include "components/signin/core/browser/profile_oauth2_token_service.h"
#include "components/signin/core/browser/signin_manager.h"
#include "components/version_info/version_info.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/storage_partition.h"
#include "net/url_request/url_request_context_getter.h"

#if defined(OS_ANDROID)
#include "chrome/browser/android/chrome_feature_list.h"
#include "chrome/browser/android/ntp/ntp_snippets_launcher.h"
#include "chrome/browser/android/offline_pages/offline_page_model_factory.h"
#include "components/ntp_snippets/offline_pages/offline_page_suggestions_provider.h"
#include "components/ntp_snippets/physical_web_pages/physical_web_page_suggestions_provider.h"
#include "components/offline_pages/offline_page_model.h"

using ntp_snippets::OfflinePageSuggestionsProvider;
using ntp_snippets::PhysicalWebPageSuggestionsProvider;
using offline_pages::OfflinePageModel;
using offline_pages::OfflinePageModelFactory;
#endif  // OS_ANDROID

using bookmarks::BookmarkModel;
using content::BrowserThread;
using image_fetcher::ImageFetcherImpl;
using ntp_snippets::BookmarkSuggestionsProvider;
using ntp_snippets::ContentSuggestionsService;
using ntp_snippets::NTPSnippetsDatabase;
using ntp_snippets::NTPSnippetsFetcher;
using ntp_snippets::NTPSnippetsService;
using ntp_snippets::NTPSnippetsScheduler;
using ntp_snippets::NTPSnippetsStatusService;
using suggestions::ImageDecoderImpl;
using suggestions::SuggestionsService;
using suggestions::SuggestionsServiceFactory;

namespace {

// Clear the tasks that can be scheduled by running services.
void ClearScheduledTasks() {
#if defined(OS_ANDROID)
  NTPSnippetsLauncher::Get()->Unschedule();
#endif  // OS_ANDROID
}

}  // namespace

// static
ContentSuggestionsServiceFactory*
ContentSuggestionsServiceFactory::GetInstance() {
  return base::Singleton<ContentSuggestionsServiceFactory>::get();
}

// static
ContentSuggestionsService* ContentSuggestionsServiceFactory::GetForProfile(
    Profile* profile) {
  DCHECK(!profile->IsOffTheRecord());
  return static_cast<ContentSuggestionsService*>(
      GetInstance()->GetServiceForBrowserContext(profile, true));
}

ContentSuggestionsServiceFactory::ContentSuggestionsServiceFactory()
    : BrowserContextKeyedServiceFactory(
          "ContentSuggestionsService",
          BrowserContextDependencyManager::GetInstance()) {
  DependsOn(BookmarkModelFactory::GetInstance());
#if defined(OS_ANDROID)
  DependsOn(OfflinePageModelFactory::GetInstance());
#endif  // OS_ANDROID
  DependsOn(ProfileOAuth2TokenServiceFactory::GetInstance());
  DependsOn(SigninManagerFactory::GetInstance());
  DependsOn(SuggestionsServiceFactory::GetInstance());
}

ContentSuggestionsServiceFactory::~ContentSuggestionsServiceFactory() {}

KeyedService* ContentSuggestionsServiceFactory::BuildServiceInstanceFor(
    content::BrowserContext* context) const {
  using State = ContentSuggestionsService::State;
  Profile* profile = Profile::FromBrowserContext(context);

  // Create the ContentSuggestionsService.
  State state =
      base::FeatureList::IsEnabled(ntp_snippets::kContentSuggestionsFeature)
          ? State::ENABLED
          : State::DISABLED;
  ContentSuggestionsService* service = new ContentSuggestionsService(state);
  if (state == State::DISABLED) {
    // Since we won't initialise the services, they won't get a chance to
    // unschedule their tasks. We do it explicitly here instead.
    ClearScheduledTasks();
    return service;
  }

#if defined(OS_ANDROID)
  // Create the OfflinePageSuggestionsProvider.
  bool recent_tabs_enabled = base::FeatureList::IsEnabled(
      ntp_snippets::kRecentOfflineTabSuggestionsFeature);
  bool downloads_enabled =
      base::FeatureList::IsEnabled(ntp_snippets::kDownloadSuggestionsFeature);
  bool download_manager_ui_enabled =
      base::FeatureList::IsEnabled(chrome::android::kDownloadsUiFeature);
  if (recent_tabs_enabled || downloads_enabled) {
    OfflinePageModel* offline_page_model =
        OfflinePageModelFactory::GetForBrowserContext(profile);

    std::unique_ptr<OfflinePageSuggestionsProvider>
        offline_page_suggestions_provider =
            base::MakeUnique<OfflinePageSuggestionsProvider>(
                recent_tabs_enabled, downloads_enabled,
                download_manager_ui_enabled, service,
                service->category_factory(), offline_page_model,
                profile->GetPrefs());
    service->RegisterProvider(std::move(offline_page_suggestions_provider));
  }
#endif  // OS_ANDROID

  // Create the BookmarkSuggestionsProvider.
  if (base::FeatureList::IsEnabled(
          ntp_snippets::kBookmarkSuggestionsFeature)) {
    BookmarkModel* bookmark_model =
        BookmarkModelFactory::GetForBrowserContext(profile);
    std::unique_ptr<BookmarkSuggestionsProvider> bookmark_suggestions_provider =
        base::MakeUnique<BookmarkSuggestionsProvider>(
            service, service->category_factory(), bookmark_model,
            profile->GetPrefs());
    service->RegisterProvider(std::move(bookmark_suggestions_provider));
  }

#if defined(OS_ANDROID)
  // Create the PhysicalWebPageSuggestionsProvider.
  if (base::FeatureList::IsEnabled(
          ntp_snippets::kPhysicalWebPageSuggestionsFeature)) {
    std::unique_ptr<PhysicalWebPageSuggestionsProvider>
        physical_web_page_suggestions_provider =
            base::MakeUnique<PhysicalWebPageSuggestionsProvider>(
                service, service->category_factory());
    service->RegisterProvider(
        std::move(physical_web_page_suggestions_provider));
  }
#endif  // OS_ANDROID

  if (base::FeatureList::IsEnabled(ntp_snippets::kArticleSuggestionsFeature)) {
    // Create the NTPSnippetsService (articles provider).
    SigninManagerBase* signin_manager =
        SigninManagerFactory::GetForProfile(profile);
    OAuth2TokenService* token_service =
        ProfileOAuth2TokenServiceFactory::GetForProfile(profile);
    scoped_refptr<net::URLRequestContextGetter> request_context =
        content::BrowserContext::GetDefaultStoragePartition(context)
            ->GetURLRequestContext();
    SuggestionsService* suggestions_service =
        SuggestionsServiceFactory::GetForProfile(profile);
    NTPSnippetsScheduler* scheduler = nullptr;
#if defined(OS_ANDROID)
    scheduler = NTPSnippetsLauncher::Get();
#endif  // OS_ANDROID
    base::FilePath database_dir(
        profile->GetPath().Append(ntp_snippets::kDatabaseFolder));
    scoped_refptr<base::SequencedTaskRunner> task_runner =
        BrowserThread::GetBlockingPool()
            ->GetSequencedTaskRunnerWithShutdownBehavior(
                base::SequencedWorkerPool::GetSequenceToken(),
                base::SequencedWorkerPool::CONTINUE_ON_SHUTDOWN);
    bool is_stable_channel =
        chrome::GetChannel() == version_info::Channel::STABLE;
    std::unique_ptr<NTPSnippetsService> ntp_snippets_service =
        base::MakeUnique<NTPSnippetsService>(
            service, service->category_factory(), profile->GetPrefs(),
            suggestions_service, g_browser_process->GetApplicationLocale(),
            scheduler, base::MakeUnique<NTPSnippetsFetcher>(
                           signin_manager, token_service, request_context,
                           profile->GetPrefs(), service->category_factory(),
                           base::Bind(&safe_json::SafeJsonParser::Parse),
                           is_stable_channel),
            base::MakeUnique<ImageFetcherImpl>(
                base::MakeUnique<ImageDecoderImpl>(), request_context.get()),
            base::MakeUnique<ImageDecoderImpl>(),
            base::MakeUnique<NTPSnippetsDatabase>(database_dir, task_runner),
            base::MakeUnique<NTPSnippetsStatusService>(signin_manager,
                                                       profile->GetPrefs()));
    service->set_ntp_snippets_service(ntp_snippets_service.get());
    service->RegisterProvider(std::move(ntp_snippets_service));
  }

  return service;
}
