#!/bin/bash

# Copyright (c) 2009 The Chromium OS Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# This script takes a path to a rootfs.ext2 which was generated by
# build_image.sh and generates an image that can be used for auto
# update.

set -e

# --- BEGIN COMMON.SH BOILERPLATE ---
# Load common CrOS utilities.  Inside the chroot this file is installed in
# /usr/lib/crosutils.  Outside the chroot we find it relative to the script's
# location.
find_common_sh() {
  local common_paths=(/usr/lib/crosutils $(dirname "$(readlink -f "$0")"))
  local path

  SCRIPT_ROOT=
  for path in "${common_paths[@]}"; do
    if [ -r "${path}/common.sh" ]; then
      SCRIPT_ROOT=${path}
      break
    fi
  done
}

find_common_sh
. "${SCRIPT_ROOT}/common.sh" || (echo "Unable to load common.sh" && exit 1)
# --- END COMMON.SH BOILERPLATE ---

# Load functions designed for image processing
if ! . "${SCRIPT_ROOT}/lib/cros_image_common.sh"; then
  echo "ERROR: Cannot load required library: lib/cros_image_common.sh; Abort."
  exit 1
fi

if [ -z "$2" -o -z "$1" ] || [ "${#@}" -ne 2 -a "${#@}" -ne 3 ]; then
  echo "usage: $0 path/to/kernel_partition_img path/to/rootfs_partition_img"
  echo "    or $0 path/to/chromiumos_img kern_part_no rootfs_part_no"
  exit 1
fi

if [ "$CROS_GENERATE_UPDATE_PAYLOAD_CALLED" != "1" ]; then
  echo "WARNING:"
  echo " This script should only be called from cros_generate_update_payload"
  echo " Please run that script with --help to see how to use it."
fi

if ! image_has_command pigz; then
  (echo "WARNING:"
   echo " Your system does not have pigz (parallel gzip) installed."
   echo " COMPRESSING WILL BE VERY SLOW. It is recommended to install pigz"
   if image_has_command apt-get; then
     echo " by 'sudo apt-get install pigz'."
   elif image_has_command emerge; then
     echo  " by 'sudo emerge pigz'."
   fi) >&2
fi

if [ $(whoami) = "root" ]; then
  echo "running $0 as root which is unneccessary"
fi

# Determine the offset size, and file name of parameters
if [ -z "$3" ]; then
  # kernnel_img rootfs_img
  KPART="$1"
  ROOT_PART="$2"
  KPART_SIZE=$(stat -c%s "$KPART")
  ROOT_PART_SIZE=$(stat -c%s "$ROOT_PART")
  KPART_OFFSET=0
  KPART_SECTORS=$((KPART_SIZE / 512))
  ROOT_OFFSET=0
  ROOT_SECTORS=$((ROOT_PART_SIZE / 512))
else
  # chromiumos_img kern_part_no rootfs_part_no
  KPART="$1"
  ROOT_PART="$1"
  KPART_OFFSET="$(image_part_offset "$KPART" "$2")" ||
    image_die "cannot retieve kernel partition offset"
  KPART_SECTORS="$(image_part_size "$KPART" "$2")" ||
    image_die "cannot retieve kernel partition size"
  ROOT_OFFSET="$(image_part_offset "$ROOT_PART" "$3")" ||
    image_die "cannot retieve root partition offset"
  ROOT_SECTORS="$(image_part_size "$ROOT_PART" "$3")" ||
    image_die "cannot retieve root partition size"
  KPART_SIZE=$((KPART_SECTORS * 512))
fi

# Sanity check size.
if [ "$KPART_SIZE" -gt $((16 * 1024 * 1024)) ]; then
  echo "Kernel partition size ($KPART_SIZE bytes) greater than 16 MiB."
  echo "That's too big."
  exit 1
fi

FINAL_OUT_FILE=$(dirname "$1")/update.gz

# Update payload format:
#  [kernel_size: big-endian uint64][kernel_blob][rootfs_blob]

# Prepare kernel_size by using printf as a number like 00000000003d0900, then
# sed to convert as: \x00\x00\x00\x00\x00\x3d\x09\x00, finally echo -e to
# convert into binary.
KPART_SIZE_SIGNATURE="$(printf "%016x" "$KPART_SIZE" |
                        sed 's/\([0-9a-f][0-9a-f]\)/\\x\1/g')"

# Build the blob!
CS_AND_RET_CODES="$(
  (echo -en "$KPART_SIZE_SIGNATURE"
    echo "Compressing kernel..." >&2
    image_dump_partial_file "$KPART" "$KPART_OFFSET" "$KPART_SECTORS"
    echo "Compressing rootfs..." >&2
    image_dump_partial_file "$ROOT_PART" "$ROOT_OFFSET" "$ROOT_SECTORS") |
  image_gzip_compress -c -9 |
  tee "$FINAL_OUT_FILE" |
  openssl sha1 -binary |
  openssl base64 |
  tr '\n' ' '
  echo ${PIPESTATUS[*]})"

EXPECTED_RET_CODES="0 0 0 0 0 0"
set -- $CS_AND_RET_CODES
CALC_CS="$1"
shift
RET_CODES="$@"
if [ "$RET_CODES" != "$EXPECTED_RET_CODES" ]; then
  echo compression/hash failed. $RET_CODES
  exit 1
fi

echo Success. hash is "$CALC_CS"
